 /**************************************************************************************************
 * SMAC Interface header file
 * 
 * Freescale Semiconductor Inc.
 * (c) Copyright 2004-2010 Freescale Semiconductor, Inc.
 * ALL RIGHTS RESERVED.
 *
***************************************************************************************************
 *
 * THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
***********************************************************************************************//*!
**************************************************************************************************/

#ifndef SMAC_INTERFACE_H_
#define SMAC_INTERFACE_H_

/************************************************************************************
*************************************************************************************
* Includes
*************************************************************************************
************************************************************************************/
#include "EmbeddedTypes.h"
#include "PhyPib.h"
#include "TransceiverDrv.h"
#include "PhyPib.h"
#include "Interrupt.h"
#include "Phy.h"                   
#include "PhyPibExtended.h"

/************************************************************************************
*************************************************************************************
* Interface macro definitions 
*************************************************************************************
************************************************************************************/

#define gSmacHeaderBytes_c					   ( 2 )            // u8DataLength + u8DestAddress bytes
#define gMaxSmacSDULength_c            (gPhyMaxDataLength_c -4 )
#define gMinSmacSDULength_c			       (gPhyMinDataLength_c)
#define gMaxOutputPower_c			         ( 0x1F )
#define gEnablePA2_Boost_c             ( 0x20 )
#define gEnablePA1_Boost_c             ( 0x40 )
#define gEnablePABoth_Boost_c          ( 0x60 )
#define gDisablePA_Boost_c             ( 0x80 )
#define gBroadcastAddress_c			       ( 0xFF )
#define gSmacBuffersSize_c             ( 280 )

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/
 extern bool_t smacStandalone;          

/************************************************************************************
*************************************************************************************
* Interface Type definitions
*************************************************************************************
************************************************************************************/
typedef struct smacPdu_tag
{
	uint8_t u8DestAddress;
	uint8_t smacPdu[1];
        //  uint8_t smacPdu[3];           //@CMA tEST
}smacPdu_t;

typedef struct txPacket_tag
{
  uint8_t u8DataLength;
  smacPdu_t smacPdu;
}txPacket_t;

typedef enum rxStatus_tag
{
  rxInitStatus,
  rxProcessingReceptionStatus_c,
  rxSuccessStatus_c, 
  rxTimeOutStatus_c,
  rxAbortedStatus_c,
  rxMaxStatus_c
} rxStatus_t;

typedef struct rxPacket_tag
{
  uint8_t    u8MaxDataLength;
  rxStatus_t rxStatus;
  uint8_t    u8DataLength;
  smacPdu_t  smacPdu;
}rxPacket_t;

typedef enum txStatus_tag
{
  txSuccessStatus_c, 
  txFailureStatus_c,
  txMaxStatus_c
} txStatus_t;

typedef enum smacErrors_tag
{
  gErrorNoError_c = 0,
  gErrorBusy_c,
  gErrorOutOfRange_c,
  gErrorNoResourcesAvailable_c,
  gErrorNoValidCondition_c,
  gErrorCorrupted_c,
  gErrorMaxError_c
} smacErrors_t;

typedef enum smacRFModes_tag
{
	gRFMode0_c = gPhyMode0_c,
	gRFMode1_c = gPhyMode1_c,
	gRFMode2_c = gPhyMode2_c,
	gRFMode3_c = gPhyMode3_c,
	gRFMaxMode_c
} smacRFModes_t;

typedef enum smacFrequencyBands_tag
{
  gSMAC_863_870MHz_c = g863__870MHz_c, /* 863-870   (Europe) */ 
  gSMAC_902_928MHz_c = g902__928MHz_c, /* 902-928   (US)     */ 
  gSMAC_950_958MHz_c = g950__958MHz_c, /* 950-958 (Japan)    */
}smacFrequencyBands_t;


typedef struct smacPacket_tag
{
	uint8_t u8SyncWordSize;
	uint8_t *u8SyncWordValue;
	uint16_t u16PreambleLength;
}smacPacket_t;

typedef struct packetConfig_tag
{	
uint16_t u16PreambleSize;
uint8_t u8SyncWordSize;
uint8_t *pu8SyncWord;
} packetConfig_t;

typedef enum scanModes_tag 
{ 
gScanModeCCA_c = 0, 
gScanModeED_c,
gMaxScanMode_c 
} scanModes_t;

typedef enum lnaGainValues_tag
{
 gLnaGain_0_c = 0,
 gLnaGain_1_c,
 gLnaGain_2_c,
 gLnaGain_3_c,
 gLnaGain_4_c,
 gLnaGain_5_c,
 gLnaGain_6_c,
 gLnaGain_7_c,	
}lnaGainValues_t;

typedef enum clkoFrequency_tag 
{
gClko32MHz_c = 0, 
gClko16MHz_c, 
gClko8MHz_c,
gClko4MHz_c,
gClko2MHz_c,
gClko1MHz_c,
gClkoOutOfRange_c
} clkoFrequency_t;

typedef enum versionedEntity_tag 
{ 
gSwPhyVersion_c, 
gSwSmacVersion_c,
gHwIcVersion_c,
gMaxVersionedEntity_c 
} versionedEntity_t;

typedef uint8_t channels_t;

typedef struct smacRFConstants_tag  
{
  uint32_t smacFirstChannelFrequency;
  uint32_t smacChannelSpacing;
  uint16_t smacTotalNumChannels;
  uint16_t smacBitRateReg;
  uint16_t smacFdevReg;
  uint16_t smacNumRssiMeasurements;
  uint8_t  smacRxBwReg;
  uint8_t  smacRxBwAfcReg;
  uint8_t  smacModulationParam;
  uint8_t  smacCcaThreshold;
} smacRFConstants_t;

typedef enum listenResolution_tag
{
gListenRes_01_c = 0x10,
gListenRes_02_c = 0x20,
gListenRes_03_c = 0x30
}listenResolution_t;


/************************************************************************************
*************************************************************************************
* External Prototypes
*************************************************************************************
************************************************************************************/

/***********************************************************************************/
/******************************** SMAC Data primitives *****************************/
/***********************************************************************************/

/************************************************************************************
* MCPSDataRequest
* 
* This data primitive is used to send an over the air packet. This is an asyncro	nous 
* function,  it means it ask SMAC to transmit one OTA packet,  but when the function 
* returns it is not sent already.
*
* Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used. 
*
* Return value:  
*   gErrorNoError_c: Everything is ok and the transmission will be performed.
*   gErrorOutOfRange_c: One of the members in the pTxMessage structure is out of 
*                      range (no valid bufer size or data buffer pointer is NULL)
*   gErrorNoResourcesAvailable_c: the radio is performing another action.
*   gErrorNoValidCondition_c: The SMAC has not been initialized 
*
************************************************************************************/
extern smacErrors_t MCPSDataRequest(txPacket_t *psTxPacket);


/************************************************************************************
* MLMERXEnableRequest
* 
* Function used to place the radio into receive mode
* 
* Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
*    
* Arguments:
* 
*        rxPacket_t *gsRxPacket: Pointer to the structure where the reception results will be stored.
*        uint32_t u32Timeout: 32-bit timeout value, this is directly the value that is stored on the MCUs timer configuration
*        
*  Return Value:
*		gErrorNoError_c: Everything is ok and the reception will be performed.
*		gErrorOutOfRange_c: One of the members in the pRxMessage structure is out of range (no valid bufer size or data buffer pointer is NULL).
*		gErrorBusy_c: the radio is performing another action.
*		gErrorNoValidCondition_c: The SMAC has not been initialized.
*************************************************************************************/
extern smacErrors_t MLMERXEnableRequest(rxPacket_t *gsRxPacket, uint32_t u32Timeout);


/************************************************************************************
* MLMERXDisableRequest
* 
* Returns the radio to idle mode from receive mode.
* 
*   Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*   Arguments: 
*   None 
*   
*   Return Value:
*      gErrorNoValidCondition_c If the Radio is not in Rx state.
*      gErrorNoError_c When the message was aborted or disabled.
*************************************************************************************/
extern smacErrors_t MLMERXDisableRequest(void);

/************************************************************************************
* MLMELinkQuality
* 
*  This function returns an integer value that is the link quality from the last received 
*  packet of the form: dBm = (-Link Quality/2).
* 
*   Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*   Arguments: 
*   None 
*   
*   Return Value:
*   uint8_t: 8 bit value representing the link quality value in dBm.
*   Returns the result in smacLastDataRxParams.linkQuality.
*************************************************************************************/
extern uint8_t  MLMELinkQuality(void);

/************************************************************************************
* MLMESetInterPacketRxDelay
* 
* This sets the inter packet delay for the packet handler 
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*   uint8_t u8InterPacketRxDelay: interpacket delay in ms
*   
*  Return Value:
*      gErrorNoValidCondition_c If the Radio is not in Rx state.
*      gErrorNoError_c When the message was aborted or disabled.
*
*************************************************************************************/
extern smacErrors_t MLMESetInterPacketRxDelay(uint8_t u8InterPacketRxDelay);





/***********************************************************************************/
/******************************** SMAC Radio primitives ****************************/
/***********************************************************************************/

/************************************************************************************
* MLMESetChannelRequest
*
* This sets the frequency on which the radio will transmit or receive on.
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*    channels_t newChannel: channel to be set
*   
*  Return Value:
*   gErrorNoError_c: The channel set has been performed
*   gErrorOutOfRange_c : channel requested is not valid
*   gErrorBusy_c: when SMAC is busy in other radio activity as transmitting or receiving data. Or performing a channel scan.
*************************************************************************************/
extern smacErrors_t MLMESetChannelRequest (channels_t newChannel);


/************************************************************************************
* MLMEGetChannelRequest
* 
* This function returns the current channel, if an error is detected it returns gChannelOutOfRange_c.
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*   None 
*   
*  Return Value:
*  channels_t (uint8_t): The current RF channel.
*  gChannelOutOfRange_c: If current channel could not be detected
*************************************************************************************/
extern channels_t   MLMEGetChannelRequest(void);


/************************************************************************************
* MLMERssi
*
* This call starts an energy detect (ED) cycle and returns the energy value (-power/2) 
* via the returned argument. For example, if the Energy Detect returns 80 then the 
* interpreted value is -80/2 or -40 dBm
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*   None 
*   
*  Return Value:
*  uint8_t: An unsigned 8-bit value representing the energy on the current channel.
*************************************************************************************/
extern uint8_t      MLMERssi(void);

/************************************************************************************
* MLMEPacketConfig
*
* This function sets the following parameters for OTA packets in radios registers:
* Preamble size, synchronization word size, and synchronization word value.
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*   packetConfig_t *pPacketCfg
*   
*  Return Value:
*  gErrorBusy_c: when SMAC is busy in other radio activity as transmitting or receiving data. 
*  Or performing a channel scan.
*  gErrorNoError_c: the packet has been configured 
*************************************************************************************/
extern smacErrors_t MLMEPacketConfig(packetConfig_t *pPacketCfg);

/************************************************************************************
* MLMERadioInit
* 
* This function initializes the Radio parameters.
* 
*  Interface assumptions:
*   None
* 
*  Arguments: 
*   None 
*   
*  Return Value:
*  gErrorNoError_c: the Radio initialization has been done successfully
*************************************************************************************/
extern smacErrors_t MLMERadioInit(void);

/************************************************************************************
* MLMEPhySoftReset
*
* This function performs a software reset on the radio, PHY and SMAC state machines.
* 
*  Interface assumptions:
*   None
* 
*  Arguments: 
*   None 
*   
*  Return Value:
*  gErrorNoError_c: If the action is performed.
*************************************************************************************/
extern smacErrors_t MLMEPhySoftReset(void);

/************************************************************************************
* MLMESetClockRate
*
* This function is called to set the desired clock out from radio.
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*   clkoFrequency_t Freq: An enumeration value that represents radio clock out frequency (CLKO)
*   
*  Return Value:
*  gErrorOutOfRange_c: If Freq exceeds the maximum clock out frequency value.
*  gErrorNoError_c:  If the action was performed correctly.
*************************************************************************************/
extern smacErrors_t MLMESetClockRate(clkoFrequency_t clkFreq);

/************************************************************************************
* MLMEStandbyRequest
*
* This primitive gets the radio in standby mode.
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*   None 
*   
*  Return Value:
*  gErrorNoError_c:  If the action was performed correctly.
*  gErrorBusy_c:     If SMAC is busy.
*************************************************************************************/
extern smacErrors_t MLMEStandbyRequest(void);

/************************************************************************************
* MLMESleepRequest
*
* This primitive causes the transceiver to go into sleep mode.
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*   None 
*   
*  Return Value:
*  gErrorNoError_c:  If the action was performed correctly.
*  gErrorBusy_c:     If SMAC is busy.
*************************************************************************************/
extern smacErrors_t MLMESleepRequest(void);

/************************************************************************************
* MLMEListenRequest
* This primitive gets the radio in listen mode.
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*   None 
*   
*  Return Value:
*  gErrorNoError_c:  If the action was performed correctly.
*  gErrorBusy_c:     If SMAC is busy.
*************************************************************************************/
extern smacErrors_t MLMEListenRequest(listenResolution_t listenResolution, uint8_t listenCoef);

/************************************************************************************
* MLMEWakeRequest
*
* This primitive wakes the radio up
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*   None 
*   
*  Return Value:
*   gErrorNoError_c:  If the action was performed correctly.
*   gErrorBusy_c:     If SMAC is busy.  
*************************************************************************************/
extern smacErrors_t MLMEWakeRequest(void);

/************************************************************************************
* MLMELnaGainAdjust
*  This function adjusts the LNA gain 
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*   lnaGainValues_t gainValue: LNA valid gain values 
*   
*  Return Value:
*   gErrorNoError_c:  If the action was performed correctly.
*   gErrorBusy_c:     If SMAC is busy. 
*************************************************************************************/
extern smacErrors_t MLMELnaGainAdjust (lnaGainValues_t gainValue);

/************************************************************************************
* MLMEDisablePABoost
*
*  This function disables the PA modes and sets it to PA0(normal).
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*   None 
*   
*  Return Value:
*   gErrorNoError_c:  If the action was performed correctly.
*   gErrorBusy_c:     If SMAC is busy. 
*************************************************************************************/
extern smacErrors_t MLMEDisablePABoost(void);

/************************************************************************************
* MLMEEnablePABoost
*
* This function enables the PA1 and PA2 or both 
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*   uint8_t u8PABoostCfg: parameter to enable PA1 and/or PA2
*   
*  Return Value:
*   gErrorNoError_c:  If the action was performed correctly.
*   gErrorBusy_c:     If SMAC is busy. 
*************************************************************************************/
extern smacErrors_t MLMEEnablePABoost(uint8_t u8PABoostCfg);

/************************************************************************************
* MLMEPAOutputAdjust
*
*
*************************************************************************************/
extern smacErrors_t MLMEPAOutputAdjust(uint8_t u8PaValue);

/************************************************************************************
* MLMESetPreambleLength
*
*
*************************************************************************************/
extern smacErrors_t MLMESetPreambleLength(uint16_t u16preambleLength);

/************************************************************************************
* MLMESetSyncWordValue
*
*
*************************************************************************************/
extern smacErrors_t MLMESetSyncWordValue(uint8_t *u8syncWordValue);

/************************************************************************************
* MLMESetSyncWordSize
*
*
*************************************************************************************/
extern smacErrors_t MLMESetSyncWordSize(uint8_t u8syncWordSize);

/************************************************************************************
* MLMESetFreqBand
* 
*  
*
************************************************************************************/
extern smacErrors_t MLMESetFreqBand(smacFrequencyBands_t freqBand, smacRFModes_t phyMode);

/************************************************************************************
*XCVRContReset
*
* This function asserts the reset line of the transceiver.
* 
*  Interface assumptions:
*   None.
* 
*  Arguments: 
*   None 
*   
*  Return Value:
*  None
*************************************************************************************/
void XCVRContReset(void);

/************************************************************************************
* XCVRRestart
*
* This function deasserts the reset line.
* 
*  Interface assumptions:
*   None
* 
*  Arguments: 
*   None 
*   
*  Return Value:
*  None
*************************************************************************************/
void XCVRRestart(void);

/***********************************************************************************/
/***************************** SMAC Management primitives **************************/
/***********************************************************************************/

/************************************************************************************
* MLMEGetRficVersion
*
* This function is used to read the version number of the software modules inside the SMAC platform.
* 
*  Interface assumptions:
*   None
* 
*  Arguments: 
*    versionedEntity_t entity: phy or smac version 
*    uint8_t *Buffer: to return either smac or phy version
*   
*  Return Value:
*   gErrorNoError_c:  If the action was performed correctly.
*   gErrorBusy_c:     If SMAC is busy. 
*************************************************************************************/
extern smacErrors_t MLMEGetRficVersion(versionedEntity_t entity, uint8_t *Buffer);

/************************************************************************************
* MLMEScanRequest
*
*  This function scans the channel passed as parameter using CCA or ED mode
*  returns the RSSI in that channel.
* 
*  Interface assumptions:
*   The SMAC and radio driver have been initialized and are ready to be used.
* 
*  Arguments: 
*   channels_t u8ChannelToScan: channel to scan 
 *  uint8_t *u8ChannelScanResult: to return the RSSI value 
*   
*  Return Value:
*   gErrorNoError_c:  If the action was performed correctly.
*   gErrorBusy_c:     If SMAC is busy. 
*************************************************************************************/
extern smacErrors_t MLMEScanRequest(channels_t u8ChannelToScan, uint8_t *u8ChannelScanResult);

/***********************************************************************************/
/******************************** SMAC MCU primitives ******************************/
/***********************************************************************************/

/************************************************************************************
* SMACDisableInterrupts
*
* This function disables the interrupts that are used by SMAC.
* 
*  Interface assumptions:
*   None
* 
*  Arguments: 
*   None 
*   
*  Return Value:
*  None
*************************************************************************************/
void SMACDisableInterrupts(void);


/************************************************************************************
* SMACEnableInterrupts
*
* This function enables the interrupts that are used by SMAC.
* 
*  Interface assumptions:
*   None 
*   
*  Arguments: 
*   None 
*   
*  Return Value:
*  None
*************************************************************************************/
void SMACEnableInterrupts(void);

#endif /* SMAC_INTERFACE_H_ */
